/* ***************************************************** **
   ch13_ordered_subsets_with_rolling_sums.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 13.
   Zbiór uporządkowany za pomocą sumy kroczącej
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

set pagesize 80
set linesize 80
set sqlformat ansiconsole

alter session set nls_date_format = 'YYYY-MM-DD';

/* -----------------------------------------------------
   Przykładowy kod do rozdziału 13.
   ----------------------------------------------------- */

-- Listing 13.2. Pobieranie danych zamówienia używanego w kolejnych przykładach

select
   c.id           as c_id
 , c.name         as c_name
 , o.id           as o_id
 , ol.product_id  as p_id
 , p.name         as p_name
 , ol.qty
from orders o
join orderlines ol
   on ol.order_id = o.id
join products p
   on p.id = ol.product_id
join customers c
   on c.id = o.customer_id
where o.id = 421
order by o.id, ol.product_id;

-- Listing 13.3. Prawdopodobne produkty do pobrania z magazynu — podane w kolejności daty zakupu

select
   i.product_id as p_id
 , ol.qty       as ord_q
 , i.qty        as loc_q
 , sum(i.qty) over (
      partition by i.product_id
      order by i.purchased, i.qty
      rows between unbounded preceding and current row
   )            as acc_q
 , i.purchased
 , i.warehouse  as wh
 , i.aisle      as ai
 , i.position   as pos
from orderlines ol
join inventory_with_dims i
   on i.product_id = ol.product_id
where ol.order_id = 421
order by i.product_id, i.purchased, i.qty;

-- Listing 13.4. Filtrowanie danych według skumulowanej sumy

select *
from (
   select
      i.product_id as p_id
    , ol.qty       as ord_q
    , i.qty        as loc_q
    , sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and current row
      )            as acc_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderlines ol
   join inventory_with_dims i
      on i.product_id = ol.product_id
   where ol.order_id = 421
)
where acc_q <= ord_q
order by p_id, purchased, loc_q;

-- Listing 13.5. Zakumulowana suma jedynie poprzednich rekordów

select
   i.product_id as p_id
 , ol.qty       as ord_q
 , i.qty        as loc_q
 , sum(i.qty) over (
      partition by i.product_id
      order by i.purchased, i.qty
      rows between unbounded preceding and 1 preceding
   )            as acc_prv_q
 , i.purchased
 , i.warehouse  as wh
 , i.aisle      as ai
 , i.position   as pos
from orderlines ol
join inventory_with_dims i
   on i.product_id = ol.product_id
where ol.order_id = 421
order by i.product_id, i.purchased, i.qty;

-- Listing 13.6. Filtrowanie według zakumulowanej sumy poprzednich rekordów

select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
from (
   select
      i.product_id as p_id
    , ol.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderlines ol
   join inventory_with_dims i
      on i.product_id = ol.product_id
   where ol.order_id = 421
)
where acc_prv_q < ord_q
order by wh, ai, pos;

-- Zmiana reguły FIFO na położenia znajdujące się najbliżej punktu początkowego, aby pracownik pokonywał jak najkrótszą drogę

select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
from (
   select
      i.product_id as p_id
    , ol.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.warehouse, i.aisle, i.position    -- << Jedyny zmieniony wiersz
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderlines ol
   join inventory_with_dims i
      on i.product_id = ol.product_id
   where ol.order_id = 421
)
where acc_prv_q < ord_q
order by wh, ai, pos;

-- Ewentualnie można skorzystać z reguły określającej, że produkty mają być pobrane z najmniejszej liczby miejsc

select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
from (
   select
      i.product_id as p_id
    , ol.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.qty desc    -- << Jedyny zmieniony wiersz
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderlines ol
   join inventory_with_dims i
      on i.product_id = ol.product_id
   where ol.order_id = 421
)
where acc_prv_q < ord_q
order by wh, ai, pos;

-- Reguła, zgodnie z którą produkty będą pobierane z tych miejsc, aby zwolnić powierzchnię dla nowych dostaw produktów

select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
from (
   select
      i.product_id as p_id
    , ol.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.qty    -- << Jedyny zmieniony wiersz
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderlines ol
   join inventory_with_dims i
      on i.product_id = ol.product_id
   where ol.order_id = 421
)
where acc_prv_q < ord_q
order by wh, ai, pos;

-- Listing 13.7. Kolejne numery przypisywane odwiedzanym alejkom w magazynie

select
   wh, ai
 , dense_rank() over (
      order by wh, ai
   ) as ai#
 , pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
from (
   select
      i.product_id as p_id
    , ol.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderlines ol
   join inventory_with_dims i
      on i.product_id = ol.product_id
   where ol.order_id = 421
)
where acc_prv_q < ord_q
order by wh, ai, pos;

-- Listing 13.8. Naprzemienna kolejność rosnąca i malejąca

select *
from (
   select
      wh, ai
    , dense_rank() over (
         order by wh, ai
      ) as ai#
    , pos, p_id
    , least(loc_q, ord_q - acc_prv_q) as pick_q
   from (
      select
         i.product_id as p_id
       , ol.qty       as ord_q
       , i.qty        as loc_q
       , nvl(sum(i.qty) over (
            partition by i.product_id
            order by i.purchased, i.qty
            rows between unbounded preceding and 1 preceding
         ), 0)        as acc_prv_q
       , i.purchased
       , i.warehouse  as wh
       , i.aisle      as ai
       , i.position   as pos
      from orderlines ol
      join inventory_with_dims i
         on i.product_id = ol.product_id
      where ol.order_id = 421
   )
   where acc_prv_q < ord_q
)
order by
   wh, ai#
 , case
      when mod(ai#, 2) = 1 then +pos
                           else -pos
   end;

-- Listing 13.9. Zerowanie numerowania alejek w każdym magazynie

select *
from (
   select
      wh, ai
    , dense_rank() over (
         partition by wh
         order by ai
      ) as ai#
    , pos, p_id
    , least(loc_q, ord_q - acc_prv_q) as pick_q
   from (
      select
         i.product_id as p_id
       , ol.qty       as ord_q
       , i.qty        as loc_q
       , nvl(sum(i.qty) over (
            partition by i.product_id
            order by i.purchased, i.qty
            rows between unbounded preceding and 1 preceding
         ), 0)        as acc_prv_q
       , i.purchased
       , i.warehouse  as wh
       , i.aisle      as ai
       , i.position   as pos
      from orderlines ol
      join inventory_with_dims i
         on i.product_id = ol.product_id
      where ol.order_id = 421
   )
   where acc_prv_q < ord_q
)
order by
   wh, ai#
 , case
      when mod(ai#, 2) = 1 then +pos
                           else -pos
   end;

-- Pracownik ma możliwość jednoczesnego kompletowania wielu zamówień, gdy porusza się po magazynach

select
   c.id           as c_id
 , c.name         as c_name
 , o.id           as o_id
 , ol.product_id  as p_id
 , p.name         as p_name
 , ol.qty
from orders o
join orderlines ol
   on ol.order_id = o.id
join products p
   on p.id = ol.product_id
join customers c
   on c.id = o.customer_id
where o.id in (422, 423)
order by o.id, ol.product_id;

-- Listing 13.10. Stosująca kolejność FIFO metoda obliczania ilości całkowitych produktów

with orderbatch as (
   select
      ol.product_id
    , sum(ol.qty) as qty
   from orderlines ol
   where ol.order_id in (422, 423)
   group by ol.product_id
)
select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
from (
   select
      i.product_id as p_id
    , ob.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderbatch ob
   join inventory_with_dims i
      on i.product_id = ob.product_id
)
where acc_prv_q < ord_q
order by wh, ai, pos;

-- Listing 13.11. Przedziały ilościowe dla każdego pobieranego produktu

with orderbatch as (
   select
      ol.product_id
    , sum(ol.qty) as qty
   from orderlines ol
   where ol.order_id in (422, 423)
   group by ol.product_id
)
select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
 , acc_prv_q + 1       as from_q
 , least(acc_q, ord_q) as to_q
from (
   select
      i.product_id as p_id
    , ob.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and current row
      ), 0)        as acc_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderbatch ob
   join inventory_with_dims i
      on i.product_id = ob.product_id
)
where acc_prv_q < ord_q
order by p_id, purchased, loc_q, wh, ai, pos;

-- Listing 13.12. Obliczanie przedziałów ilościowych z użyciem tylko jednej funkcji analitycznej

with orderbatch as (
   select
      ol.product_id
    , sum(ol.qty) as qty
   from orderlines ol
   where ol.order_id in (422, 423)
   group by ol.product_id
)
select
   wh, ai, pos, p_id
 , least(loc_q, ord_q - acc_prv_q) as pick_q
 , acc_prv_q + 1                   as from_q
 , least(acc_prv_q + loc_q, ord_q) as to_q
from (
   select
      i.product_id as p_id
    , ob.qty       as ord_q
    , i.qty        as loc_q
    , nvl(sum(i.qty) over (
         partition by i.product_id
         order by i.purchased, i.qty
         rows between unbounded preceding and 1 preceding
      ), 0)        as acc_prv_q
    , i.purchased
    , i.warehouse  as wh
    , i.aisle      as ai
    , i.position   as pos
   from orderbatch ob
   join inventory_with_dims i
      on i.product_id = ob.product_id
)
where acc_prv_q < ord_q
order by p_id, purchased, loc_q, wh, ai, pos;

-- Listing 13.13. Obliczanie przedziałów ilościowych dla zamówień

select
   ol.order_id    as o_id
 , ol.product_id  as p_id
 , ol.qty
 , nvl(sum(ol.qty) over (
      partition by ol.product_id
      order by ol.order_id
      rows between unbounded preceding and 1 preceding
   ), 0) + 1      as from_q
 , nvl(sum(ol.qty) over (
      partition by ol.product_id
      order by ol.order_id
      rows between unbounded preceding and 1 preceding
   ), 0) + ol.qty as to_q
from orderlines ol
where ol.order_id in (422, 423)
order by ol.product_id, ol.order_id;

-- Listing 13.14. Złączenie nakładające przedziały ilościowe dla produktów i zamówień

with olines as (
   select
      ol.order_id    as o_id
    , ol.product_id  as p_id
    , ol.qty
    , nvl(sum(ol.qty) over (
         partition by ol.product_id
         order by ol.order_id
         rows between unbounded preceding and 1 preceding
      ), 0) + 1      as from_q
    , nvl(sum(ol.qty) over (
         partition by ol.product_id
         order by ol.order_id
         rows between unbounded preceding and 1 preceding
      ), 0) + ol.qty as to_q
   from orderlines ol
   where ol.order_id in (422, 423)
), orderbatch as (
   select
      ol.p_id
    , sum(ol.qty) as qty
   from olines ol
   group by ol.p_id
), fifo as (
   select
      wh, ai, pos, p_id, loc_q
    , least(loc_q, ord_q - acc_prv_q) as pick_q
    , acc_prv_q + 1                   as from_q
    , least(acc_prv_q + loc_q, ord_q) as to_q
   from (
      select
         i.product_id as p_id
       , ob.qty       as ord_q
       , i.qty        as loc_q
       , nvl(sum(i.qty) over (
            partition by i.product_id
            order by i.purchased, i.qty
            rows between unbounded preceding and 1 preceding
         ), 0)        as acc_prv_q
       , i.purchased
       , i.warehouse  as wh
       , i.aisle      as ai
       , i.position   as pos
      from orderbatch ob
      join inventory_with_dims i
         on i.product_id = ob.p_id
   )
   where acc_prv_q < ord_q
)
select
   f.wh, f.ai, f.pos, f.p_id
 , f.pick_q, f.from_q as p_f_q, f.to_q as p_t_q
 , o.o_id  , o.from_q as o_f_q, o.to_q as o_t_q
from fifo f
join olines o
   on o.p_id = f.p_id
   and o.to_q >= f.from_q
   and o.from_q <= f.to_q
order by f.p_id, f.from_q, o.from_q;

-- Listing 13.15. Ilości produktów trafiające do poszczególnych zamówień

with olines as (
   select
      ol.order_id    as o_id
    , ol.product_id  as p_id
    , ol.qty
    , nvl(sum(ol.qty) over (
         partition by ol.product_id
         order by ol.order_id
         rows between unbounded preceding and 1 preceding
      ), 0) + 1      as from_q
    , nvl(sum(ol.qty) over (
         partition by ol.product_id
         order by ol.order_id
         rows between unbounded preceding and 1 preceding
      ), 0) + ol.qty as to_q
   from orderlines ol
   where ol.order_id in (422, 423)
), orderbatch as (
   select
      ol.p_id
    , sum(ol.qty) as qty
   from olines ol
   group by ol.p_id
), fifo as (
   select
      wh, ai, pos, p_id, loc_q
    , least(loc_q, ord_q - acc_prv_q) as pick_q
    , acc_prv_q + 1                   as from_q
    , least(acc_prv_q + loc_q, ord_q) as to_q
   from (
      select
         i.product_id as p_id
       , ob.qty       as ord_q
       , i.qty        as loc_q
       , nvl(sum(i.qty) over (
            partition by i.product_id
            order by i.purchased, i.qty
            rows between unbounded preceding and 1 preceding
         ), 0)        as acc_prv_q
       , i.purchased
       , i.warehouse  as wh
       , i.aisle      as ai
       , i.position   as pos
      from orderbatch ob
      join inventory_with_dims i
         on i.product_id = ob.p_id
   )
   where acc_prv_q < ord_q
)
select
   f.wh, f.ai, f.pos, f.p_id
 , f.pick_q, o.o_id
 , least(
      f.loc_q
    , least(o.to_q, f.to_q) - greatest(o.from_q, f.from_q) + 1
   ) as q_f_o
from fifo f
join olines o
   on o.p_id = f.p_id
   and o.to_q >= f.from_q
   and o.from_q <= f.to_q
order by f.p_id, f.from_q, o.from_q;

-- Listing 13.16. Ostateczna postać kodu SQL pozwalającego stosować kolejność FIFO podczas pobierania produktów dla wielu zamówień

with olines as (
   select
      ol.order_id    as o_id
    , ol.product_id  as p_id
    , ol.qty
    , nvl(sum(ol.qty) over (
         partition by ol.product_id
         order by ol.order_id
         rows between unbounded preceding and 1 preceding
      ), 0) + 1      as from_q
    , nvl(sum(ol.qty) over (
         partition by ol.product_id
         order by ol.order_id
         rows between unbounded preceding and 1 preceding
      ), 0) + ol.qty as to_q
   from orderlines ol
   where ol.order_id in (422, 423)
), orderbatch as (
   select
      ol.p_id
    , sum(ol.qty) as qty
   from olines ol
   group by ol.p_id
), fifo as (
   select
      wh, ai, pos, p_id, loc_q
    , least(loc_q, ord_q - acc_prv_q) as pick_q
    , acc_prv_q + 1                   as from_q
    , least(acc_prv_q + loc_q, ord_q) as to_q
   from (
      select
         i.product_id as p_id
       , ob.qty       as ord_q
       , i.qty        as loc_q
       , nvl(sum(i.qty) over (
            partition by i.product_id
            order by i.purchased, i.qty
            rows between unbounded preceding and 1 preceding
         ), 0)        as acc_prv_q
       , i.purchased
       , i.warehouse  as wh
       , i.aisle      as ai
       , i.position   as pos
      from orderbatch ob
      join inventory_with_dims i
         on i.product_id = ob.p_id
   )
   where acc_prv_q < ord_q
), pick as (
   select
      f.wh, f.ai
    , dense_rank() over (
         order by wh, ai
      ) as ai#
    , f.pos, f.p_id
    , f.pick_q, o.o_id
    , least(
         f.loc_q
       , least(o.to_q, f.to_q) - greatest(o.from_q, f.from_q) + 1
      ) as q_f_o
   from fifo f
   join olines o
      on o.p_id = f.p_id
      and o.to_q >= f.from_q
      and o.from_q <= f.to_q
)
select
   p.wh, p.ai, p.pos
 , p.p_id, p.pick_q
 , p.o_id, p.q_f_o
from pick p
order by p.wh
       , p.ai#
       , case
            when mod(p.ai#, 2) = 1 then +p.pos
                                   else -p.pos
         end;

/* ***************************************************** */
